package ga.view.factory;

import com.jme3.asset.AssetManager;
import com.jme3.input.InputManager;
import com.jme3.math.Vector3f;
import com.jme3.post.FilterPostProcessor;
import com.jme3.post.filters.LightScatteringFilter;
import com.jme3.post.ssao.SSAOFilter;
import com.jme3.renderer.ViewPort;
import com.jme3.shadow.PssmShadowRenderer;
import com.jme3.shadow.PssmShadowRenderer.CompareMode;
import com.jme3.shadow.PssmShadowRenderer.FilterMode;
import com.jme3.system.AppSettings;

/**
 * A factory for creating effects and adding them to scenes.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public final class EffectsFactory {

  /**
   * Instantiation is not allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private EffectsFactory() {
  }

  /**
   * Adds a scene processor that renders shadows.
   * 
   * @param assetManager
   *          The asset manager.
   * @param settings
   *          The app settings.
   * @param viewPort
   *          The viewport.
   * @param lightLoc
   *          The location of the light.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void addShadowProcessor(final AssetManager assetManager,
      final AppSettings settings, final ViewPort viewPort,
      final Vector3f lightLoc) {

    final boolean shadowsEnabled = settings.getBoolean("UseShadows");

    if (shadowsEnabled) {
      final Vector3f shadowDir = lightLoc.negate().normalizeLocal();
      final PssmShadowRenderer pssmRenderer = new PssmShadowRenderer(
          assetManager, 1024, 6);
      pssmRenderer.setDirection(shadowDir);
      pssmRenderer.setLambda(0.1f);
      pssmRenderer.setShadowIntensity(0.3f);
      pssmRenderer.setCompareMode(CompareMode.Hardware);
      pssmRenderer.setFilterMode(FilterMode.PCF8);
      // pssmRenderer.displayDebug();

      viewPort.addProcessor(pssmRenderer);
    }
  }

  /**
   * Adds an SSAO processor that renders some kind of global illumination.
   * 
   * @param assetManager
   *          The asset manager.
   * @param inputManager
   *          The input manager.
   * @param settings
   *          The app settings.
   * @param viewPort
   *          The viewport.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void addSSAOProcessor(final AssetManager assetManager,
      final InputManager inputManager, final AppSettings settings,
      final ViewPort viewPort) {

    final boolean ssaoEnabled = settings.getBoolean("UseSSAO");

    if (ssaoEnabled) {
      final FilterPostProcessor fpp = new FilterPostProcessor(assetManager);
      SSAOFilter ssaoFilter;

      ssaoFilter = new SSAOFilter(1.1f, 3.4f, 3.9f, 0.4f);
      fpp.addFilter(ssaoFilter);

      // if (inputManager != null) {
      // final SSAOUI ui = new SSAOUI(inputManager, ssaoFilter);
      // }

      viewPort.addProcessor(fpp);
    }
  }

  /**
   * Adds a processor that creates light scattering effects.
   * 
   * @param assetManager
   *          The asset manager.
   * @param inputManager
   *          The input manager.
   * @param settings
   *          The app settings.
   * @param viewPort
   *          The viewport.
   * @param lightLoc
   *          The light location.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void addLightScatteringProcessor(
      final AssetManager assetManager, final InputManager inputManager,
      final AppSettings settings, final ViewPort viewPort,
      final Vector3f lightLoc) {

    final boolean lightScatteringEnabled = settings
        .getBoolean("UseLightScattering");

    if (lightScatteringEnabled) {
      final FilterPostProcessor fpp = new FilterPostProcessor(assetManager);
      final LightScatteringFilter filter = new LightScatteringFilter(lightLoc);
      // filter.setNbSamples(80);
      // filter.setLightDensity(0.2f);
      // filter.setBlurStart(.3f);
      fpp.addFilter(filter);

      // if (inputManager != null) {
      // final LightScatteringUI ui = new LightScatteringUI(inputManager,
      // filter);
      // }

      viewPort.addProcessor(fpp);
    }

  }
}
